#!/usr/bin/env python3
# coding: utf-8

import os
import sys
import subprocess
import datetime
import shutil
import time
import lzma
import paramiko
import configparser
import glob
from ping3 import ping

version = "1.3"

# 🔧 Charger la configuration depuis config.ini
config = configparser.ConfigParser()
config.read("config.ini")

cfg = {
    "DATA": config["GENERAL"]["DATA"],
    "FIC_SRC": config["GENERAL"]["FIC_SRC"],
    "DELAY": int(config["GENERAL"]["DELAY"]),
    "SIZE_LIMIT": int(config["GENERAL"]["SIZE_LIMIT"]),
    "SFTP_SRV": config["SFTP"]["SFTP_SRV"],
    "SFTP_PORT": int(config["SFTP"]["SFTP_PORT"]),
    "SFTP_USER": config["USER"]["TAG"],
    "SFTP_PWD": config["USER"]["PWD"],
    "SFTP_DIR": config["SFTP"]["SFTP_DIR"],
}

#  Création du répertoire DATA s'il n'existe pas
os.makedirs(cfg["DATA"], exist_ok=True)

# Adapter le chemin du fichier source
FIC_SRC = os.path.join(cfg["DATA"], cfg["FIC_SRC"])

def sftp_connect():
    print("📡 Connecting SFTP server...")
    transport = paramiko.Transport((cfg["SFTP_SRV"], cfg["SFTP_PORT"]))
    transport.connect(username=cfg["SFTP_USER"], password=cfg["SFTP_PWD"])
    sftp = paramiko.SFTPClient.from_transport(transport)
    return sftp, transport


def is_reachable(host):
    return ping(host) 

def compresser_xz(source):
    """Compresse le fichier en .xz"""
    archive_name = source + ".xz"
    with open(source, "rb") as f_in, lzma.open(archive_name, "wb") as f_out:
        shutil.copyfileobj(f_in, f_out)
    print(f"✅ Compression completed : {archive_name}")

def envoyer_fichiers_sftp():
    """Envoie tous les fichiers .xz disponibles sur le serveur SFTP"""
    try:
        # Connexion SFTP
        sftp, transport = sftp_connect()
        print(f"✅ Connected")

        #  Liste et envoie les fichiers .xz du répertoire DATA
        for fichier in os.listdir(cfg["DATA"]):
            if fichier.endswith(".xz"):
                local_path = os.path.join(cfg["DATA"], fichier)
                remote_filename = str(int(time.time())) + '.' + fichier
                remote_path = f"{cfg['SFTP_DIR']}/{remote_filename}"

                print(f"⬆️  Sending {fichier}")
                # print (cfg["SFTP_SRV"], cfg["SFTP_PORT"], cfg["SFTP_USER"], cfg["SFTP_PWD"])
                print ("📦 local:", fichier)
                print ("📤 remote:", remote_filename)

                try:
                    sftp.put(local_path, remote_path+".part")
                    sftp.rename(remote_path+".part", remote_path)
                    print(f"✅ Transfer successful : {fichier}")
                    os.remove(local_path)  # Suppression après succès
                except Exception as e:
                    print(f"❌ Sending failed {fichier} : {e}")

        print("✅ Connexion closed")
        sftp.close()
        transport.close()
    
    except Exception as e:
        print(f"❌ Error SFTP : {e}")


# Fonction pour vérifier la mise à jour et télécharger le fichier
def check_update():
    try:
        # Connexion SFTP
        sftp, transport = sftp_connect()
        print(f"✅ Connected")

        remote_path = "download/joli_compagnon_latest"
        
        # Vérification si le fichier existe
        try:
            sftp.stat(remote_path)  # Tester si le fichier existe
            local_path = os.path.join(cfg["DATA"], "joli_compagnon_latest.py")
            sftp.get(remote_path, local_path)  # Télécharger le fichier
            print(f"✅ Update downloaded")
        except FileNotFoundError:
            print(f"✅ No update available (but that's OK...)")

        print("✅ Connexion closed")
        sftp.close()
        transport.close()

    except Exception as e:
        print(f"❌ Connection or download failed : {e}")



def flush_stream():
            # 📅 Générer le timestamp
        timestamp = datetime.datetime.now().strftime("%Y%m%d-%H%M%S")
        fic_dest = os.path.join(cfg["DATA"], f"nmea.{timestamp}.{cfg['SFTP_USER']}")

        # deplacer le fichier source
        shutil.move(FIC_SRC, fic_dest)

        # Compression en .xz
        compresser_xz(fic_dest)

        # Suppression du fichier original après compression
        os.remove(fic_dest)

        print(f"✅ {fic_dest}.xz ready to be sent")



# Liste des variables essentielles à vérifier
required_keys = ["FIC_SRC", "DELAY", "SIZE_LIMIT", "SFTP_SRV", "SFTP_PORT", "SFTP_USER", "SFTP_DIR", "SFTP_PWD"]

# Vérification des variables essentielles
for key in required_keys:
    if not cfg.get(key):
        print(f"❌ ERROR: {key} missing in config.ini")
        sys.exit(1)

####################################################################
# DEBUT
####################################################################


# Test d'une mise à jour

print ("🧐 Version:", version)
check_update()

while True:
    
    file_ok = os.path.exists(FIC_SRC)

    ping_ok = is_reachable(cfg["SFTP_SRV"])


    #  Vérification de la taille du fichier
    file_size_ok = False
    if file_ok:
        file_size_ok = os.path.getsize(FIC_SRC) > cfg["SIZE_LIMIT"]
    
    pending_files_ok = glob.glob(os.path.join(cfg["DATA"], "nmea*.xz")) 

    if file_ok and (ping_ok or file_size_ok):
        flush_stream()


    pending_files_ok = glob.glob(os.path.join(cfg["DATA"], "nmea*.xz"))

    #  Si le ping est OK, envoyer les fichiers via SFTP
    if pending_files_ok and ping_ok:
        envoyer_fichiers_sftp()

    #  Attente avant la prochaine vérification
    print ("⏳ Waiting", cfg["DELAY"] , "secondes before next batch...")
    time.sleep(cfg["DELAY"])

