#!/usr/bin/env python3
# coding: utf-8

import serial
import configparser
import os
import time

# === LECTURE CONFIG ===
config = configparser.ConfigParser(inline_comment_prefixes=('#', ';'))
config.read("config.ini")

cfg = {
    "DATA": config["GENERAL"]["DATA"],
    "FIC_SRC": config["GENERAL"]["FIC_SRC"],
    "DELAY": int(config["GENERAL"]["DELAY"]),
    "SIZE_LIMIT": int(config["GENERAL"]["SIZE_LIMIT"]),
    "SFTP_SRV": config["SFTP"]["SFTP_SRV"],
    "SFTP_PORT": int(config["SFTP"]["SFTP_PORT"]),
    "SFTP_USER": config["USER"]["TAG"],
    "SFTP_PWD": config["USER"]["PWD"],
    "SFTP_DIR": config["SFTP"]["SFTP_DIR"],
    "GPS_PORT": config["GPS"]["GPS_PORT"],
    "GPS_BAUDS": int(config["GPS"]["GPS_BAUDS"]),
    "GPS_INTERVAL": int(config["GPS"]["GPS_INTERVAL"]),
}

PORT = cfg["GPS_PORT"]
BAUD = cfg["GPS_BAUDS"]
OUTPUT_FILE = os.path.join(cfg["DATA"], cfg["FIC_SRC"])

TRAMES_USED = {}
TRAMES_SUFFIX = ("RMC", "GGA", "VTG", "GLL", "MDA", "MWV", "MTW", "MMB", "XDR")
FLUSH_INTERVAL = 15  # écrit toutes les X secondes pour éviter corruption

def run_gps():
    buffer = []
    last_flush_time = time.time()

    print(f"📡 Listening to GPS on {PORT} at {BAUD} baud")

    while True:
        try:
            # Ouvre le port série
            with serial.Serial(PORT, BAUD, timeout=1) as ser:
                print(f"✅ Connected to GPS ({PORT})")

                while True:
                    line = ser.readline().decode(errors="ignore").strip()
                    if not line:
                        continue

                    if line.startswith("$"):
                        sentence_type = line[1:6]
                        if sentence_type in TRAMES_USED or sentence_type.endswith(TRAMES_SUFFIX):
                            print("✅ :", line)
                            buffer.append(line + "\n")
                        else:
                            print("❌ :", line)

                    # Écriture périodique du buffer
                    if time.time() - last_flush_time >= FLUSH_INTERVAL and buffer:
                        try:
                            with open(OUTPUT_FILE, "a", encoding="utf-8") as f:
                                f.writelines(buffer)
                                f.flush()
                            print(f"💾 : {len(buffer)} frames saved {OUTPUT_FILE}")
                        except Exception as e:
                            print(f"⚠️ Erreur d’écriture : {e}")
                        finally:
                            buffer.clear()
                            last_flush_time = time.time()

                    # Pause optionnelle (GPS_INTERVAL)
                    if cfg["GPS_INTERVAL"] > 0:
                        time.sleep(cfg["GPS_INTERVAL"])

        except serial.SerialException as e:
            print(f"❌ GPS disconnected or not found: {e}")
            print("⏳ Retrying in 30 seconds...")
            time.sleep(30)
        except Exception as e:
            print(f"⚠️ Unexpected error: {e}")
            time.sleep(10)

if __name__ == "__main__":
    try:
        run_gps()
    except KeyboardInterrupt:
        print("\n🛑 GPS reader manually stopped")

