#!/usr/bin/env python3
# coding: utf-8


import socket
import re
import configparser
import os
import time

config = configparser.ConfigParser()
config.read("config.ini")

cfg = {
    "DATA": config["GENERAL"]["DATA"],
    "FIC_SRC": config["GENERAL"]["FIC_SRC"],
    "DELAY": int(config["GENERAL"]["DELAY"]),
    "SIZE_LIMIT": int(config["GENERAL"]["SIZE_LIMIT"]),
    "SFTP_SRV": config["SFTP"]["SFTP_SRV"],
    "SFTP_PORT": int(config["SFTP"]["SFTP_PORT"]),
    "SFTP_USER": config["USER"]["TAG"],
    "SFTP_PWD": config["USER"]["PWD"],
    "SFTP_DIR": config["SFTP"]["SFTP_DIR"],
    "NMEA_SRV": config["NMEA"]["NMEA_SRV"],
    "NMEA_PORT": int(config["NMEA"]["NMEA_PORT"]),
}

HOST = cfg["NMEA_SRV"]
PORT = cfg["NMEA_PORT"]
OUTPUT_FILE = os.path.join(cfg["DATA"], cfg["FIC_SRC"])

TRAMES_REGEX = re.compile(r"^\$(..)(RMC|GNS|GGA|GLL|MWV|MDA|MBP|MTW|MTA|HDG|VHW|DPT|DBT|RPM),")

FLUSH_INTERVAL = 15  # ecrit tt les x secondes pour ne pas garder de desc ouverts

def run_client():
    buffer = []
    last_flush_time = time.time()

    while True:
        try:
            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
                print(f"🔌 Trying to connect to {HOST}:{PORT}...")
                s.connect((HOST, PORT))
                print(f"📡 Connected to {HOST}:{PORT} via TCP")

                while True:
                    data = s.recv(1024)
                    if not data:
                        print("⚠️ Connection lost or closed by server")
                        break

                    lines = data.decode("ascii", errors="ignore").strip().split("\r\n")

                    for line in lines:
                        if TRAMES_REGEX.match(line):
                            print("✅ :", line)
                            buffer.append(line.rstrip("\r\n") + "\n")
                        else:
                            print("❌ :", line)

                    if time.time() - last_flush_time >= FLUSH_INTERVAL and buffer:
                        with open(OUTPUT_FILE, "a") as f:
                            f.writelines(buffer)
                            f.flush()
                            buffer.clear()
                            last_flush_time = time.time()

        except (ConnectionRefusedError, OSError) as e:
            print(f"❌ Connection failed or lost: {e}")

        print("⏳ Reconnecting in 60 seconds...")
        time.sleep(60)

if __name__ == "__main__":
    try:
        run_client()
    except KeyboardInterrupt:
        print("\n🛑 TCP client manually stopped")

